using System;
using System.Collections.Generic;
using System.Text;
using System.Threading;

namespace SIPPBXv3
{
    public interface ISIPPBXPluginHost
    {
        GTAPIASM.GTAPIChan Channel { get;set;}
        SIPPBXChan PBX_Channel { get;set;}
        SIPPBXBase PBXBase { get;set;}
        VoiceMailBox VMB { get;set;}

        bool Register(ISIPPBXPluginClient ipi);
        string DisplayMenu(string audio_fn, int maxDigits, string termStr, int timeOut, bool stopPlayingForFirstKey);
        string DisplayMenuEx(List<string> audio_files, int maxDigits, string termStr, int timeOut, bool stopPlayingForFirstKey);
        string DetectDTMF(int maxDigits, string termStr, int timeOut);
        string PlayAudio(string audio_fn, int maxDigits, string termStr, int timeOut);
        string PlayAudioEx(List<string> audio_files, int maxDigits, string termStr, int timeOut);
        string RecordAudio(string audio_fn, int maxDigits, string termStr, int timeOut);
        int HangUp();
        void WriteLog(string logInfo);

        void StartPBX();
        void StopPBX();

        //some useful functions to re-route the call.
        bool ToExtension(string exten_no);
        bool ToIVRMenu(string menu_name);
        bool ToHuntGroup(string acd_name, bool set_front);
        bool ToMonitorGroup(string mg_name);
        bool ToRingGroup(string rg_name);
        bool ToConferenceRoom(string conf_name);
        //bool ToConferenceRoomEx(string conf_name, int opt);
        bool ToVoiceMailBox(string exten_no);
        bool ToPlugin(string plugin_name);
        bool ToNumber(string number, SIPAccount sip_acct);
        bool ToNumberEx(string number, SIPAccount sip_acct, ref int maxSec);
        bool ToSIPServer(string destSIPIP, ushort destSIPPort, string caller, string callee, string auFile, ref int maxSec);

        //other call control command
        bool DisconnectExtension(string exten_no);
        bool SetChanRunPlugin(int ch, string plugin_name);
        bool ResetChannel(int ch);
        bool ResetACDGroup(string acd_group_name);
        bool AgentLogin(string agent_code, string exten_no, string acd_name);
        bool AgentLogout(string agent_code, string exten_no, string acd_name);
        bool AgentPaused(string agent_code, int pause); //1 = pause, 0 = un-pause

        //conference room control
        bool SetChanInConferenceRoom(int ch, string conf_name, int opt); //opt: 0 = take out of conference room. 1 = add into conference room. 2 = monitor(listening only, not speaking)
        
		void SetChanConferenceBitMask(int ch, uint bitMask);
		
		SIPConferRoom GetConferenceRoomByName(string conf_name);
		IntPtr GetConferenceRoomHandle(string conf_name);
		int GetConferenceRoomIndex(string conf_name);
		
		SIPConferRoom CreateConferenceRoom(string conf_name);
		void DestroyConferenceRoom(string conf_name);

        //Access sharing object between channels
        object GetUserObj(int idx);
        bool SetUserObj(int idx, object obj);

        //Access channel sharing object
        object GetChanUserObj(int ch, int idx);
        bool SetChanUserObj(int ch, int idx, object obj);

    }

    public interface ISIPPBXPluginClient : ICloneable
    {
        string Name { get;set;}
        string Type { get;set;}
        ISIPPBXPluginHost Host { get;set;}
        void Start();
        void Done();
    }

    public class SIPPBXPluginHost : ISIPPBXPluginHost
    {
        public SIPPBXChan pbx_chan;
        public GTSIPPBXEnv env;
        public Thread client_thread;
        public EventWaitHandle event_wait_handle;
        public VoiceMailBox vmb;

        public ISIPPBXPluginClient m_Client;

        public const string RESULT_DISCONNECTED = "Disconnected";
        public const string RESULT_ERROR = "Error";

        public GTAPIASM.GTAPIChan Channel 
        {
            get { return env.GetChannel(pbx_chan.index); }
            set { }
        }

        public SIPPBXChan PBX_Channel
        {
            get { return pbx_chan; }
            set { }
        }


        public SIPPBXBase PBXBase 
        {
            get { return env.pbx; }
            set { }
        }

        public VoiceMailBox VMB
        {
            get { return vmb; }
            set { }
        }

        public ISIPPBXPluginClient Client 
        { 
            get{return m_Client;}
            set { m_Client = value;}
        }

        static void Run(object obj)
        {
            SIPPBXPluginHost host = (SIPPBXPluginHost)obj;
            host.event_wait_handle.WaitOne();

            try
            {
                host.m_Client.Start();
                host.m_Client.Done();
            }
            catch (Exception ex)
            {
                host.WriteLog(ex.ToString());
            }
        }

        public SIPPBXPluginHost()
        {
            pbx_chan = null;
            env = null;
            client_thread = new Thread(Run);
            event_wait_handle = new EventWaitHandle(false, EventResetMode.AutoReset);
            vmb = null;
        }

        public bool Register(ISIPPBXPluginClient ipi)
        {
            try
            {
                m_Client = (ISIPPBXPluginClient)ipi.Clone();
            }
            catch (Exception ex)
            {
                WriteLog(ex.ToString());
                return false;
            }
            m_Client.Host = this;
            client_thread.Start(this);

            return true;
        }

        public string DisplayMenuEx(List<string> audio_files, int maxDigits, string termStr, int timeOut, bool stopPlayingForFirstKey)
        {
            GTOpPluginCompound op = null;

            try
            {
                lock (env)
                {
                    op = (GTOpPluginCompound)pbx_chan.async_op_compound;
                    pbx_chan.DTMFBuf = "";

                    if (op != null)
                    {
                        op._op_audio_play_ex = new GTOpAudioPlayEx(op, env, pbx_chan, pbx_chan.DTMFBuf, audio_files, new List<string>(), maxDigits, termStr, timeOut, stopPlayingForFirstKey);
                        op._op_audio_play_ex.perform();
                    }
                    else
                        return RESULT_ERROR;
                }
            }
            catch (Exception ex)
            {
                env.LOG_Trace(1, ex.ToString());
                env.LogoutText(ex.ToString());
            }

            event_wait_handle.WaitOne();

            GTAPIASM.GTAPIChan api_chan = env.GetChannel(pbx_chan.index);

            if (api_chan.ch_status == GTAPIASM.GTAPI_CHANNEL_STATE.IDLE)
                return RESULT_DISCONNECTED;
            else
            {
                if (op != null)
                    return op._op_audio_play_ex.getDTMFStr();
                else
                    return RESULT_ERROR;
            }
        }

        public string DisplayMenu(string audio_fn, int maxDigits, string termStr, int timeOut, bool stopPlayingForFirstKey)
        {
            GTOpPluginCompound op = null;

            try
            {
                lock (env)
                {
                    op = (GTOpPluginCompound)pbx_chan.async_op_compound;
                    pbx_chan.DTMFBuf = "";
                    List<string> audio_files = new List<string>();
                    audio_files.Add(audio_fn);

                    if (op != null)
                    {
                        op._op_audio_play_ex = new GTOpAudioPlayEx(op, env, pbx_chan, pbx_chan.DTMFBuf, audio_files, new List<string>(), maxDigits, termStr, timeOut, stopPlayingForFirstKey);
                        op._op_audio_play_ex.perform();
                    }
                    else
                        return RESULT_ERROR;
                }
            }
            catch (Exception ex)
            {
                env.LOG_Trace(1, ex.ToString());
                env.LogoutText(ex.ToString());
            }

            event_wait_handle.WaitOne();

            GTAPIASM.GTAPIChan api_chan = env.GetChannel(pbx_chan.index);

            if (api_chan.ch_status == GTAPIASM.GTAPI_CHANNEL_STATE.IDLE)
                return RESULT_DISCONNECTED;
            else
            {
                if (op != null)
                    return op._op_audio_play_ex.getDTMFStr();
                else
                    return RESULT_ERROR;
            }
        }

        public string DetectDTMF(int maxDigits, string termStr, int timeOut)
        {
            GTOpPluginCompound op = null;

            try
            {
                lock (env)
                {
                    op = (GTOpPluginCompound)pbx_chan.async_op_compound;
                    pbx_chan.DTMFBuf = "";

                    if (op != null)
                    {
                        op._op_dtmf_detect = new GTOpDTMFDetect(op, env, pbx_chan, pbx_chan.DTMFBuf, new List<string>(), maxDigits, termStr, timeOut);
                        op._op_dtmf_detect.perform();
                    }
                    else
                        return RESULT_ERROR;
                }
            }
            catch (Exception ex)
            {
                env.LOG_Trace(1, ex.ToString());
                env.LogoutText(ex.ToString());
            }

            event_wait_handle.WaitOne();

            GTAPIASM.GTAPIChan api_chan = env.GetChannel(pbx_chan.index);

            if (api_chan.ch_status == GTAPIASM.GTAPI_CHANNEL_STATE.IDLE)
                return RESULT_DISCONNECTED;
            else
            {
                if (op != null)
                    return op._op_dtmf_detect.getDTMFStr();
                else
                    return RESULT_ERROR;
            }
        }

        public string PlayAudioEx(List<string> audio_files, int maxDigits, string termStr, int timeOut)
        {
            GTOpPluginCompound op = null;

            try
            {
                lock (env)
                {
                    op = (GTOpPluginCompound)pbx_chan.async_op_compound;
                    pbx_chan.DTMFBuf = "";

                    if (op != null)
                    {
                        op._op_audio_play = new GTOpAudioPlay(op, env, pbx_chan, pbx_chan.DTMFBuf, audio_files, new List<string>(), maxDigits, termStr, timeOut);
                        op._op_audio_play.perform();
                    }
                    else
                        return RESULT_ERROR;
                }
            }
            catch (Exception ex)
            {
                env.LOG_Trace(1, ex.ToString());
                env.LogoutText(ex.ToString());
            }

            event_wait_handle.WaitOne();

            GTAPIASM.GTAPIChan api_chan = env.GetChannel(pbx_chan.index);

            if (api_chan.ch_status == GTAPIASM.GTAPI_CHANNEL_STATE.IDLE)
                return RESULT_DISCONNECTED;
            else
            {
                if (op != null)
                    return op._op_audio_play.getDTMFStr();
                else
                    return RESULT_ERROR;
            }
        }

        public string PlayAudio(string audio_fn, int maxDigits, string termStr, int timeOut)
        {
            GTOpPluginCompound op = null;

            try
            {
                lock (env)
                {
                    op = (GTOpPluginCompound)pbx_chan.async_op_compound;
                    pbx_chan.DTMFBuf = "";
                    List<string> audio_files = new List<string>();
                    audio_files.Add(audio_fn);

                    if (op != null)
                    {
                        op._op_audio_play = new GTOpAudioPlay(op, env, pbx_chan, pbx_chan.DTMFBuf, audio_files, new List<string>(), maxDigits, termStr, timeOut);
                        op._op_audio_play.perform();
                    }
                    else
                        return RESULT_ERROR;
                }
            }
            catch (Exception ex)
            {
                env.LOG_Trace(1, ex.ToString());
                env.LogoutText(ex.ToString());
            }

            event_wait_handle.WaitOne();

            GTAPIASM.GTAPIChan api_chan = env.GetChannel(pbx_chan.index);

            if (api_chan.ch_status == GTAPIASM.GTAPI_CHANNEL_STATE.IDLE)
                return RESULT_DISCONNECTED;
            else
            {
                if (op != null)
                    return op._op_audio_play.getDTMFStr();
                else
                    return RESULT_ERROR;
            }
        }

        public string RecordAudio(string audio_fn, int maxDigits, string termStr, int timeOut)
        {
            GTOpPluginCompound op = null;

            try
            {
                lock (env)
                {
                    op = (GTOpPluginCompound)pbx_chan.async_op_compound;
                    pbx_chan.DTMFBuf = "";
                    if (op != null)
                    {
                        op._op_audio_record = new GTOpAudioRecord(op, env, pbx_chan, audio_fn, maxDigits, termStr, timeOut);
                        op._op_audio_record.perform();
                    }
                    else
                        return RESULT_ERROR;
                }
            }
            catch (Exception ex)
            {
                env.LOG_Trace(1, ex.ToString());
                env.LogoutText(ex.ToString());
            }

            event_wait_handle.WaitOne();

            GTAPIASM.GTAPIChan api_chan = env.GetChannel(pbx_chan.index);

            if (api_chan.ch_status == GTAPIASM.GTAPI_CHANNEL_STATE.IDLE)
                return RESULT_DISCONNECTED;
            else
            {
                if (op != null)
                    return op._op_audio_record.getDTMFStr();
                else
                    return RESULT_ERROR;
            }
        }

        public int HangUp()
        {
            if (env.IsChanConnected(pbx_chan.index))
                env.DisconnectCall(pbx_chan.index, 0, "", "PBX: plugin hangup");

            return 0;
        }

        public int HangUpEx(int code, string reason)
        {
            if (env.IsChanConnected(pbx_chan.index))
                env.DisconnectCall(pbx_chan.index, code, reason, "PBX: plugin hangupEx");

            return 0;
        }

        public object GetUserObj(int idx)
        {
            return env.GetUserObj(idx);
        }

        public bool SetUserObj(int idx, object obj)
        {
            return env.SetUserObj(idx, obj);
        }

        public object GetChanUserObj(int ch, int idx)
        {
            return env.GetChanUserObj(ch, idx);
        }

        public bool SetChanUserObj(int ch, int idx, object obj)
        {
            return env.SetChanUserObj(ch, idx, obj);
        }

        public void WriteLog(string logInfo)
        {
            string sInfo = "Plugin(" + Client.Name + ") Trace: " + logInfo;
            //env.LOG_Trace(4, sInfo); LogoutText will put the log info on screen and pbx log both.
            env.LogoutText(sInfo);
        }

        public void StartPBX()
        {
            env.pbxMain.StartPBX();
        }

        public void StopPBX()
        {
            env.pbxMain.StopPBX();
        }

        public bool transferCallEx(string callee, string caller, SIPPBXExten extn, SIPAccount acct, ref int maxSec)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;
            GTOpPluginCompound op = (GTOpPluginCompound)pbx_chan.async_op_compound;

            if (maxSec <= 0) return false;

            CallLimit _cl = new CallLimit();
            _cl.Seconds = maxSec;

            //check call limit and call block
            //CallLimit _cl = _pbx.getCallLimit(null, GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, caller), GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, callee), pbx_chan);
            //if (_cl != null)
            //{
            //    if (_cl.Seconds <= 0)
            //    {
            //        _env.LOG_Trace(4, "SIPPBXPlugin:transferCall Call is rejected because there is no enough minute left!");
            //        _env.LogoutText("SIPPBXPlugin:transferCall Call is rejected bbecause there is no enough minute left!");
            //        return false;
            //    }
            //}


            if (!_pbx.passCallBlackList(_env, null, GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, caller), GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, callee)))
            {
                _env.LOG_Trace(4, "SIPPBXPlugin:transferCall Call is rejected because it is in blacklist!");
                _env.LogoutText("SIPPBXPlugin:transferCall Call is rejected bbecause it is in blacklist!");
                return false;
            }


            SIPPBXChan chan2 = env.pbx.SeizeChannelForOutbound(env);
            //GTAPIASM.GTAPIChan api_chan = null;

            if (chan2 == null)
            {
                //not enough outbound channel available
                //just disconnect it
                //_env.Send_HungUp(_chan.index);

                _env.LogoutText("SIPPBXPluginHost::transferCall SeizeChannelForOutbound return null. No enough outbound channel for calling!");

                return false;
            }

            op._op_audio_play = null;
            op._op_audio_play_ex = null;
            op._op_audio_record = null;
            op._op_dtmf_detect = null;

            chan2.ResetAll(_chan.unique_call_id, _chan.call_dir, _pbx, extn);
            chan2.link_exten = extn;
            if (extn != null)
                if (extn.RingTimeoutSec >= 0) //as for -1, it is always forwarding. won't make call out so this state is not updated if set to 10.
                    _pbx.SetExtenCallingState(extn, _env, 10); //offered

            op._op_call_transfer = new GTOpCallTransfer(op, _env, _chan, _chan.DTMFBuf, chan2, callee, caller, (extn != null) ? extn.RingTimeoutSec : 0, acct);
            op._op_call_transfer._cl = _cl;
            op._op_call_transfer.perform();

            event_wait_handle.WaitOne();

            if (op._result == GTOpAsync.ResultCode.OP_RESULT_SUCCESS)
            {
                TimeSpan tsp = op._op_call_transfer.callDisconnectedTime - op._op_call_transfer.callConnectedTime;
                maxSec = Convert.ToInt32(tsp.TotalSeconds);
                return true;
            }

            return false;
        }

        public bool ToSIPServer(string destSIPIP, ushort destSIPPort, string caller, string callee, string auFile, ref int maxSec)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;
            GTOpPluginCompound op = (GTOpPluginCompound)pbx_chan.async_op_compound;

            if (maxSec < 0) return false;

            CallLimit _cl = null;

            if (maxSec > 0)
            {
                _cl = new CallLimit();
                _cl.Seconds = maxSec;
            }

            //check call limit and call block
            //CallLimit _cl = _pbx.getCallLimit(null, GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, caller), GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, callee), pbx_chan);
            //if (_cl != null)
            //{
            //    if (_cl.Seconds <= 0)
            //    {
            //        _env.LOG_Trace(4, "SIPPBXPlugin:transferCall Call is rejected because there is no enough minute left!");
            //        _env.LogoutText("SIPPBXPlugin:transferCall Call is rejected bbecause there is no enough minute left!");
            //        return false;
            //    }
            //}


            if (!_pbx.passCallBlackList(_env, null, GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, caller), GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, callee)))
            {
                _env.LOG_Trace(4, "SIPPBXPlugin:transferCall Call is rejected because it is in blacklist!");
                _env.LogoutText("SIPPBXPlugin:transferCall Call is rejected bbecause it is in blacklist!");
                return false;
            }


            SIPPBXChan chan2 = env.pbx.SeizeChannelForOutbound(env);
            //GTAPIASM.GTAPIChan api_chan = null;

            if (chan2 == null)
            {
                //not enough outbound channel available
                //just disconnect it
                //_env.Send_HungUp(_chan.index);

                _env.LogoutText("SIPPBXPluginHost::ToSIPServer SeizeChannelForOutbound return null. No enough outbound channel for calling!");
                _env.LOG_Trace(1, "SIPPBXPluginHost::ToSIPServer SeizeChannelForOutbound return null. No enough outbound channel for calling!");
                return false;
            }

            op._op_audio_play = null;
            op._op_audio_play_ex = null;
            op._op_audio_record = null;
            op._op_dtmf_detect = null;

            chan2.ResetAll(_chan.unique_call_id, _chan.call_dir, _pbx, null);
            chan2.link_exten = null;

            string callee_displayname = GTAPIASM.GTAPIEnv.GetSIPAddressInfo(0, callee);
            string callee_num = GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, callee);

            string newCallee = callee_displayname + "<sip:" + callee_num + "@" + destSIPIP + ":" + destSIPPort.ToString() + ">";

            op._op_call_transfer = new GTOpCallTransfer(op, _env, _chan, _chan.DTMFBuf, chan2, newCallee, caller, 0, null);
            op._op_call_transfer._cl = _cl;
            op._op_call_transfer.audio_recording_fn = auFile;
            op._op_call_transfer.disc_main_chan = true;
            op._op_call_transfer.perform();

            event_wait_handle.WaitOne();

            if (op._result == GTOpAsync.ResultCode.OP_RESULT_SUCCESS)
            {
                TimeSpan tsp = op._op_call_transfer.callDisconnectedTime - op._op_call_transfer.callConnectedTime;
                maxSec = Convert.ToInt32(tsp.TotalSeconds);
                return true;
            }

            return false;
        }

        public bool transferCall(string callee, string caller, SIPPBXExten extn, SIPAccount acct)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;
            GTOpPluginCompound op = (GTOpPluginCompound)pbx_chan.async_op_compound;
            
			//check call limit and call block
			CallLimit _cl = _pbx.getCallLimit(null, GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, caller), GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, callee), pbx_chan);
			if (_cl != null)
			{
				if (_cl.Seconds <= 0)
				{
					_env.LOG_Trace(4, "SIPPBXPlugin:transferCall Call is rejected because there is no enough minute left!");
                    _env.LogoutText("SIPPBXPlugin:transferCall Call is rejected bbecause there is no enough minute left!");
					return false;
				}
			}


			if (!_pbx.passCallBlackList(_env, null, GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, caller), GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, callee)))
			{
                _env.LOG_Trace(4, "SIPPBXPlugin:transferCall Call is rejected because it is in blacklist!");
                _env.LogoutText("SIPPBXPlugin:transferCall Call is rejected bbecause it is in blacklist!");
				return false;
			}           
            

            SIPPBXChan chan2 = env.pbx.SeizeChannelForOutbound(env);
            //GTAPIASM.GTAPIChan api_chan = null;

            if (chan2 == null)
            {
                //not enough outbound channel available
                //just disconnect it
                //_env.Send_HungUp(_chan.index);

                _env.LogoutText("SIPPBXPluginHost::transferCall SeizeChannelForOutbound return null. No enough outbound channel for calling!");

                return false;
            }

            op._op_audio_play = null;
            op._op_audio_play_ex = null;
            op._op_audio_record = null;
            op._op_dtmf_detect = null;

            chan2.ResetAll(_chan.unique_call_id, _chan.call_dir, _pbx, extn);
            chan2.link_exten = extn;
            if(extn != null)
                if (extn.RingTimeoutSec >= 0) //as for -1, it is always forwarding. won't make call out so this state is not updated if set to 10.
                    _pbx.SetExtenCallingState(extn, _env, 10); //offered

            op._op_call_transfer = new GTOpCallTransfer(op, _env, _chan, _chan.DTMFBuf, chan2, callee, caller, (extn != null) ? extn.RingTimeoutSec : 0, acct);
            op._op_call_transfer._cl = _cl;
            op._op_call_transfer.perform();

            event_wait_handle.WaitOne();

            if (op._result == GTOpAsync.ResultCode.OP_RESULT_SUCCESS)
                return true;

            return false;
        }

        public bool TryReachVoiceMailBox(string vmb_name)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;

            SIPPBXExten extn = _pbx.getExtensionByName(vmb_name);
            if (extn != null)
            {
                if (extn.vmb != null)
                {
                    _chan.async_op_compound = new GTOpVMB(_pbx, _env, _chan, extn.vmb);
                    _chan.async_op_compound.start();
                    return true;
                }
            }

            return false;
        }

        public void TryReachVoiceMailBox(VoiceMailBox vmb)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;
            _chan.async_op_compound = new GTOpVMB(_pbx, _env, _chan, vmb);
            _chan.async_op_compound.start();
        }

        public bool ToNumberEx(string number, SIPAccount sip_acct, ref int maxSec)
        {
            string destaddr;
            string fromaddr;

            if (number.Contains("<sip:") && number.Contains("@"))
            {
                destaddr = number;
            }
            else if (sip_acct != null)
            {
                destaddr = number;
                destaddr += "@" + sip_acct.DomainServer;
                destaddr = "<sip:" + destaddr + ">";
            }
            else
                return false;

            if (sip_acct == null)
            {
                fromaddr = "";
            }
            else
            {
                fromaddr = sip_acct.UserName;
                if (sip_acct.OutboundAcceptOtherID || sip_acct.SIPTrunk == 1)
                {
                    if (sip_acct.DIDList.Count > 0)
                        fromaddr = sip_acct.DIDList[0];
                }

                if (fromaddr == "")
                    fromaddr = "unknown";

                if (sip_acct.UseLocalIPInFrom)
                {
                    if (env.pbx.sip_set.sipAddr.Length > 0)
                        fromaddr += "@" + env.pbx.sip_set.sipAddr;
                    else
                        fromaddr += "@" + env.GetMappedPublicSIPIPAddress();
                }
                else
                    fromaddr += "@" + sip_acct.DomainServer;
                fromaddr = sip_acct.DisplayName + "<sip:" + fromaddr + ">";
            }

            return transferCallEx(destaddr, fromaddr, null, sip_acct, ref maxSec);
        }

        public bool ToNumber(string number, SIPAccount sip_acct)
        {
            string destaddr;
            string fromaddr;

            if (number.Contains("<sip:") && number.Contains("@"))
            {
                destaddr = number;
            }
            else if (sip_acct != null)
            {
                destaddr = number;
                destaddr += "@" + sip_acct.DomainServer;
                destaddr = "<sip:" + destaddr + ">";
            }
            else
                return false;

            if (sip_acct == null)
            {
                fromaddr = "";
            }
            else
            {
                fromaddr = sip_acct.UserName;
                if (sip_acct.OutboundAcceptOtherID || sip_acct.SIPTrunk == 1)
                {
                    if (sip_acct.DIDList.Count > 0)
                        fromaddr = sip_acct.DIDList[0];
                }

                if (fromaddr == "")
                    fromaddr = "unknown";

                if (sip_acct.UseLocalIPInFrom)
                {
                    if (env.pbx.sip_set.sipAddr.Length > 0)
                        fromaddr += "@" + env.pbx.sip_set.sipAddr;
                    else
                        fromaddr += "@" + env.GetMappedPublicSIPIPAddress();
                }
                else
                    fromaddr += "@" + sip_acct.DomainServer;
                fromaddr = sip_acct.DisplayName + "<sip:" + fromaddr + ">";
            }

            return transferCall(destaddr, fromaddr, null, sip_acct);
        }

        public bool ToExtension(string exten_no)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;

            SIPPBXExten extn = _pbx.getExtensionByName(exten_no);
            if (extn == null)
                return false;

            GTAPIASM.GTAPIChan api_chan = _env.GetChannel(_chan.index);

            string caller_num = SIPPBXWinUtil.BuildCallerID(api_chan, "");
            string caller_username = GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, caller_num);
            string called_num = GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, api_chan.callee_num); 


            if (extn.IsVirtualExten())
            {
                //calling to a virtual extension
                if (extn.VirtualExtenDestAddr.Contains(".") || extn.VirtualExtenDestAddr.Contains("@"))
                {
                    //sip address
                    string sipDestAddr = extn.VirtualExtenDestAddr.Replace("*", called_num);

                    if (sipDestAddr.Contains("sip:"))
                    {
                        if (!transferCall(sipDestAddr, caller_num, extn, null))
                        {
                            //no enough idle channel for outbound
                            if (extn.vmb != null)
                                TryReachVoiceMailBox(extn.vmb);
                            else
                                _env.DisconnectCall(_chan.index, 0, "", "PBX: no enough idle channel for outbound in PBXPlugin");
                        }
                    }
                    else
                    {
                        if (!transferCall("<sip:" + sipDestAddr + ">", caller_num, extn, null))
                        {
                            //no enough idle channel for outbound
                            if (extn.vmb != null)
                                TryReachVoiceMailBox(extn.vmb);
                            else
                                _env.DisconnectCall(_chan.index, 0, "", "PBX: no enough idle channel for outbound in PBXPlugin 1");
                        }
                    }
                }
                else
                {
                    //to see if it matchs any outbound rules
                    SIPPBXDialPlan dp1 = _pbx.getDialPlanByCalledNum(_env, extn.VirtualExtenDestAddr, "", GTAPIASM.GTAPIEnv.GetSIPAddressInfo(1, caller_num), GTAPIASM.GTAPIEnv.GetSIPAddressInfo(2, caller_num), _pbx.getExtensionBySIPAddr(caller_num), 1);
                    if (dp1 != null)
                    {
                        SIPAccount sip_acct = _pbx.getDialPlanSIPAccount(dp1);
                        if (sip_acct != null)
                        {
                            //outbound, should take out predix digit, then use another channel to dialout
                            string destaddr = dp1.OutboundPrepend + extn.VirtualExtenDestAddr.Substring(dp1.OutboundPreStrip.Length);
                            if (extn.bAcceptOtherID && called_num.Length > 0)
                                destaddr = called_num;
                            destaddr += "@" + sip_acct.DomainServer;
                            destaddr = "<sip:" + destaddr + ">";

                            string fromaddr = sip_acct.UserName;
                            if (sip_acct.OutboundAcceptOtherID || sip_acct.SIPTrunk == 1)
                            {
                                SIPPBXExten ext0 = _pbx.getExtensionBySIPAddr(caller_num);
                                if (ext0 != null)
                                {
                                    if (ext0.AlternativePhoneNumber.Length > 0)
                                    {
                                        fromaddr = ext0.AlternativePhoneNumber;
                                    }
                                    else if (sip_acct.DIDList.Count > 0)
                                    {
                                        fromaddr = sip_acct.DIDList[0];
                                        if (sip_acct.OutboundAppendExtenID)
                                        {
                                            fromaddr += ext0.UserName;
                                        }
                                    }
                                }
                                else
                                {
                                    fromaddr = caller_username;
                                    if (fromaddr == "unknown")
                                        fromaddr = sip_acct.UserName;
                                }
                            }

                            if (fromaddr == "")
                            {
                                if (caller_username.Length > 0 && (sip_acct.OutboundAcceptOtherID || sip_acct.SIPTrunk == 1))
                                    fromaddr = caller_username;
                                else if (sip_acct.DIDList.Count > 0 && (sip_acct.OutboundAcceptOtherID || sip_acct.SIPTrunk == 1))
                                    fromaddr = sip_acct.DIDList[0];
                                else if (sip_acct.UserName.Length > 0)
                                    fromaddr = sip_acct.UserName;
                                else
                                    fromaddr = "unknown";
                            }

                            if (sip_acct.UseLocalIPInFrom)
                            {
                                if (_pbx.sip_set.sipAddr.Length > 0)
                                    fromaddr += "@" + _pbx.sip_set.sipAddr;
                                else
                                    fromaddr += "@" + _env.GetMappedPublicSIPIPAddress();
                            }
                            else
                                fromaddr += "@" + sip_acct.DomainServer;
                            fromaddr = SIPPBXWinUtil.GetCallerDisplayName(api_chan) + "<sip:" + fromaddr + ">";

                            if (!transferCall(destaddr, fromaddr, extn, sip_acct))
                            {
                                //no enough idle channel for outbound
                                if (extn.vmb != null)
                                    TryReachVoiceMailBox(extn.vmb);
                                else
                                    return false;
                            }
                        }
                        else
                        {
                            //cannot find right SIP account for outbound
                            if (extn.vmb != null)
                                TryReachVoiceMailBox(extn.vmb);
                            else
                                return false;
                        }
                    }
                    else
                    {
                        if (extn.vmb != null)
                            TryReachVoiceMailBox(extn.vmb);
                        else
                            return false;
                    }
                }
            }
            else
            {
                //regular extension
                if (extn.IsRegistered())
                {
                    string sCallee = SIPPBXWinUtil.BuildCalleeID(api_chan, extn, _chan);

                    if (!transferCall(sCallee, caller_num, extn, null))
                    {
                        //no enough idle channel for outbound
                        if (extn.vmb != null)
                            TryReachVoiceMailBox(extn.vmb);
                        else
                            return false;
                    }
                }
                else
                {
                    if (extn.vmb != null)
                        TryReachVoiceMailBox(extn.vmb);
                    else
                        return false;
                }
            }

            //_exten = extn;

            return true;
        }

        public bool ToIVRMenu(string menu_name)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;

            //_ivr = _ivr.ParentIVRMenu;
            SIPPBXIVR ivrmenu = _pbx.getIVRMenuByName(menu_name);
             
            if (ivrmenu != null)
            {
                _chan.async_op_compound = new GTOpIVRMenu(_pbx, _env, _chan, ivrmenu);
                _chan.async_op_compound.start();
                return true;
            }
            else
            {
                WriteLog("Cannot find IVRMenu(" + menu_name + ") for the call on channel " + _chan.index.ToString());
                return false;
            }
        }

        public bool ToHuntGroup(string acd_name, bool set_front)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;

            SIPPBXACDHuntGroup hg = _pbx.getACDByName(acd_name);
            if (hg != null)
            {
                if (hg.calls.Count >= hg.maxNumOfCalls && hg.maxNumOfCalls > 0)
                {
                    //ACD queue full
                    if (hg.callForwardingType == 0) //to another ACD group
                    {
                        SIPPBXACDHuntGroup hg1 = _pbx.getACDByName(hg.callForwardingPlan);
                        if (hg1 != null)
                        {
                            _chan.async_op_compound = new GTOpACD(_pbx, _env, _chan, hg1, _pbx.moh_dir, set_front);
                            _chan.async_op_compound.start();
                            return true;
                        }
                        else
                        {
                            WriteLog("Cannot find huntgroup(" + hg.callForwardingPlan + ") for incoming calls.");
                            return false;
                        }
                    }
                    else if (hg.callForwardingType == 1) //to another Dialplan
                    {
                        SIPPBXDialPlan dp_hg = _pbx.getDialPlanByPlanName(hg.callForwardingPlan);
                        if (dp_hg != null)
                        {
                            _chan.dp = dp_hg;
                            _env.DoDialPlan(_chan.index, _chan, null, null);
                            return true;
                        }
                        else
                        {
                            WriteLog("Cannot find dialplan(" + hg.callForwardingPlan + ") for incoming calls.");
                            return false;
                        }
                    }
                    else
                    {
                        WriteLog("ACD group " + hg.hgName + " queue is full, but forwarding type is not correct!");
                        return false;
                    }
                }
                else
                {
                    _chan.async_op_compound = new GTOpACD(_pbx, _env, _chan, hg, _pbx.moh_dir, set_front);
                    _chan.async_op_compound.start();
                    return true;
                }

            }
            else
            {
                WriteLog("Cannot find huntgroup(" + acd_name + ") for the call on channel " + _chan.index.ToString());
                return false;
            }
        }

        public bool ToMonitorGroup(string mg_name)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;

            SIPPBXMonitorGroup mg = _pbx.getMonitorGroupByNumber(mg_name);
            if (mg != null)
            {
                _chan.async_op_compound = new GTOpMonitorGroup(_pbx, _env, _chan, mg, _chan.link_exten);
                _chan.async_op_compound.start();
                return true;
            }
            else
            {
                WriteLog("Cannot find monitor group(" + mg_name + ") for the call on channel " + _chan.index.ToString());
                return false;
            } 
        }

        public bool ToRingGroup(string rg_name)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;

            SIPPBXRingGroup rg = _pbx.getRingGroupByName(rg_name);
            if (rg != null)
            {
                _chan.async_op_compound = new GTOpRingGroup(_pbx, _env, _chan, rg, _pbx.moh_dir);
                _chan.async_op_compound.start();
                return true;
            }
            else
            {
                WriteLog("Cannot find ringgroup(" + rg_name + ") for the call on channel " + _chan.index.ToString());
                return false;
            }
        }

        public bool ToConferenceRoom(string conf_name)
        {
            return env.pbx.SetChanIntoConferenceRoom(env, pbx_chan, conf_name, 1);
        }

        public bool ToVoiceMailBox(string exten_no)
        {
            return TryReachVoiceMailBox(exten_no);
        }

        public bool ToPlugin(string plugin_name)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = pbx_chan;

            ISIPPBXPluginClient client = _pbx.GetPluginClientInterfaceByName(plugin_name);

            if (client == null)
            {
                _env.LogoutText("Plugin(" + plugin_name + ") doesn't exist!");
                return false;
            }

            if (client.Type != "IVRMenu")
            {
                _env.LogoutText("Plugin(" + plugin_name + ") is not IVRMenu Type!");
                return false;
            }

            SIPPBXPluginHost plugin_host = new SIPPBXPluginHost();
            _chan.plugin_host = plugin_host;
            plugin_host.pbx_chan = _chan;
            plugin_host.env = _env;
            if (plugin_host.Register(client))
            {
                _chan.async_op_compound = new GTOpPluginCompound(_pbx, _env, _chan);
                plugin_host.event_wait_handle.Set();
            }

            return true;

        }


        public bool DisconnectExtension(string exten_no)
        {
            for (int i = 0; i < env.GetChannelCount(); i++)
            {
                SIPPBXChan achan = env.pbx.chan_list[i];
                if (achan.link_exten != null)
                {
                    if (achan.link_exten.UserName == exten_no)
                    {
                        env.DisconnectCall(i, 0, "", "PBX: DisconnectExtension in PBXPlugin");
                        if (achan.link_chan != null)
                            env.DisconnectCall(achan.link_chan.index, 0, "", "PBX: DisconnectExtension disconnect link_chan in PBXPlugin");
                    }
                }
            }

            return true;
        }

        public bool SetChanInConferenceRoom(int ch, string conf_name, int opt)
        {
            //opt
            //0 = set it out of conference
            //1 = two way
            //2 = listen only

            return env.pbx.SetChanIntoConferenceRoom(env, env.pbx.chan_list[ch], conf_name, opt);
        }
        
        public void SetChanConferenceBitMask(int ch, uint bitMask)
        {
			env.SetChanConfMask(ch, bitMask);
		}
		
		public SIPConferRoom GetConferenceRoomByName(string conf_name)
		{
			return env.pbx.getConferenceRoomByName(conf_name);
		}
		
		public IntPtr GetConferenceRoomHandle(string conf_name)
		{
			SIPConferRoom room = env.pbx.getConferenceRoomByName(conf_name);
			
			if(room != null)
			{
				return room.conf_handle;
			}

            return IntPtr.Zero;
		}
		
		public int GetConferenceRoomIndex(string conf_name)
		{
			SIPConferRoom room = env.pbx.getConferenceRoomByName(conf_name);
			
			if(room != null)
			{
				return env.GetConfIndex(room.conf_handle);
			}			
			
			return -1;
		}
		
		public SIPConferRoom CreateConferenceRoom(string conf_name)
		{
            SIPConferRoom room = env.pbx.getConferenceRoomByName(conf_name);

            if (room == null)
            {
                room = new SIPConferRoom();
                room.conf_handle = env.CreateConf();

                room.conf_name = conf_name;
                room.conf_max_num = 30; //myReader.GetInt32(2);

                room.join_prompt = "";
                room.leave_prompt = "";

                room.moh_dir = "";
                room.disc_call = false;
                room.host_pw = "";
                room.host_pw_prompt = "";
                room.conf_record = false;

                env.pbx.sip_conferooms.Add(room);
            }
			
			return room;		
		}
		
		public void DestroyConferenceRoom(string conf_name)
		{
			SIPConferRoom room = GetConferenceRoomByName(conf_name);
			if(room != null)
			{
				env.DestroyConf(room.conf_handle);
				env.pbx.sip_conferooms.Remove(room);
			}
		}

        public bool SetChanRunPlugin(int ch, string plugin_name)
        {
            SIPPBX _pbx = env.pbx;
            GTSIPPBXEnv _env = env;
            SIPPBXChan _chan = _pbx.chan_list[ch];

            ISIPPBXPluginClient client = _pbx.GetPluginClientInterfaceByName(plugin_name);

            if (client == null)
            {
                _env.LogoutText("Plugin(" + plugin_name + ") doesn't exist!");
                return false;
            }

            if (client.Type != "IVRMenu")
            {
                _env.LogoutText("Plugin(" + plugin_name + ") is not IVRMenu Type!");
                return false;
            }

            SIPPBXPluginHost plugin_host = new SIPPBXPluginHost();
            _chan.plugin_host = plugin_host;
            plugin_host.pbx_chan = _chan;
            plugin_host.env = _env;
            if (plugin_host.Register(client))
            {
                _chan.async_op_compound = new GTOpPluginCompound(_pbx, _env, _chan);
                plugin_host.event_wait_handle.Set();
            }

            return true;
        }

        public bool ResetChannel(int ch)
        {
            env.DisconnectCall(ch, 0, "", "PBX: ResetChannel in PBXPlugin");
            return true;
        }

        public bool ResetACDGroup(string acd_group_name)
        {
            SIPPBX pbx = env.pbx;
            SIPPBXACDHuntGroup acd_group = pbx.getACDByName(acd_group_name);

            if (acd_group != null)
            {
                if (acd_group.calls.Count > 0)
                {
                    for (int j = 0; j < acd_group.calls.Count; j++)
                    {
                        GTOpAsync op = acd_group.calls[j].op;
                        if (op != null)
                        {
                            if (op.getPBXChan() != null)
                            {
                                env.DisconnectCall(op.getPBXChan().index, 0, "", "PBX: ResetACDGroup in PBXPlugin");
                            }
                        }
                    }
                }

                return true;
            }

            return false;
        }

        public bool AgentPaused(string agent_code, int pause) //1 = pause, 0 = un-pause
        {
            SIPPBXAgent agent = env.pbx.getAgentByCode(agent_code);
            if (agent == null)
                return false;

            agent.Paused = pause == 1;

            if (!SIPPBXCFGDB.UpdateAgentPausedStatusIntoDB(agent, env.pbxMain, env.pbx, env, env.pbxMain.dbPBXSet.myConn, env.pbxMain.GetPBXLog()))
                env.pbxMain.dbPBXSet.ResetConnection();

            return true;
        }

        public bool AgentLogin(string agent_code, string exten_no, string acd_name)
        {
            SIPPBX pbx = env.pbx;
            SIPPBXExten extn = pbx.getExtensionByName(exten_no);
            SIPPBXACDHuntGroup acd = pbx.getACDByName(acd_name);
            SIPPBXAgent agent = pbx.getAgentByCode(agent_code);

            if(agent== null || extn == null) return false;
            return pbx.AgentLogin(agent, extn, env, "", "", "", acd);
        }

        public bool AgentLogout(string agent_code, string exten_no, string acd_name)
        {
            SIPPBX pbx = env.pbx;
            SIPPBXExten extn = pbx.getExtensionByName(exten_no);
            SIPPBXACDHuntGroup acd = pbx.getACDByName(acd_name);
            SIPPBXAgent agent = pbx.getAgentByCode(agent_code);

            if (agent == null || extn == null) return false;
            return pbx.AgentLogout(agent, extn, env, "", "", "", acd);
        }
    }

}
